// signal_link.hpp

// Copyright Stjepan Rajko 2007. Use, modification and
// distribution is subject to the Boost Software License, Version
// 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#ifndef AME_SIGNAL_LINK_HPP
#define AME_SIGNAL_LINK_HPP

#define BOOST_SIGNALS_STATIC_LINK
#include <boost/signal.hpp>
#include <boost/bind.hpp>
#include <boost/preprocessor/punctuation/comma.hpp>
#include <boost/preprocessor/repetition/repeat.hpp>

namespace ame
{

template<typename Descendant, typename Signature>
class signal_link;

#define SIGNAL_LINK_ARG(z,n,text) BOOST_PP_COMMA() _##n
#define SIGNAL_LINK_ARGS(n) BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_ADD(n,1),SIGNAL_LINK_ARG,_)

#define SIGNAL_LINK_CAST(n) typename boost::function_traits<Signature>::arg##n##_type
#define SIGNAL_LINK_CAST_COMMA(z,n,text) SIGNAL_LINK_CAST(n) BOOST_PP_COMMA()
#define SIGNAL_LINK_CASTS(n) BOOST_PP_REPEAT_FROM_TO(1,n,SIGNAL_LINK_CAST_COMMA,_) BOOST_PP_IF(n,SIGNAL_LINK_CAST(n),BOOST_PP_EMPTY())

namespace detail
{
	template<typename T, int Arity, typename Signature>
	struct slot_type_impl;
	
	#define SIGNAL_LINK_SLOT_TYPE_TEMPLATE(_arity) \
	template<typename T, typename Signature> \
	struct slot_type_impl<T, _arity, Signature> \
	{ \
	typedef typename boost::function_traits<Signature>::result_type (T::*type)(SIGNAL_LINK_CASTS(_arity)); \
	};

	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(0)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(1)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(2)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(3)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(4)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(5)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(6)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(7)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(8)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(9)
	SIGNAL_LINK_SLOT_TYPE_TEMPLATE(10)

	// Constructs a type corresponding to pointer to member of T with signature Signature.
	// e.g. slot_type<some_class, void(float)>::type is void (some_class::*) (float)
	template<typename T, typename Signature>
	struct slot_type
		: public slot_type_impl<T, boost::function_traits<Signature>::arity, Signature>
	{};
}

template<typename T, typename Signature>
struct slot_selector_t
{
	T &link;
	typename detail::slot_type<T, Signature>::type func;

	slot_selector_t<T, Signature>(T &link, typename detail::slot_type<T, Signature>::type func)
		: link(link), func(func) {}
	operator T &() {return link;}
};

///	Allows functions other than operator() to serve as signal_link slots.
/**	\ingroup Utility
	\sa signal_link for an \ref signal_link_multi_in_example "example"
*/
template<typename Signature, typename T>
slot_selector_t<T, Signature> slot_selector(T &link, typename detail::slot_type<T, Signature>::type func)
{
	return slot_selector_t<T, Signature>(link, func);
}

namespace detail
{
	template<typename T, int Arity, typename Signature>
	struct signal_link_connect_impl;
	
#define SIGNAL_LINK_CONNECT_TEMPLATE(_arity) \
	template<typename T, typename Signature> \
	struct signal_link_connect_impl<T, _arity, Signature> \
	{ \
		static void connect(boost::signal<Signature> &signal, T &link) \
		{ \
			signal.connect(boost::bind( \
			(typename slot_type<T, Signature>::type) &T::operator(), \
				boost::ref(link) SIGNAL_LINK_ARGS(_arity))); \
		} \
		static void connect_slot(boost::signal<Signature> &signal, const slot_selector_t<T, Signature> &slot) \
		{ \
			signal.connect(boost::bind( \
			(typename slot_type<T, Signature>::type) slot.func, \
				boost::ref(slot.link) SIGNAL_LINK_ARGS(_arity))); \
		} \
	};

	SIGNAL_LINK_CONNECT_TEMPLATE(0);
	SIGNAL_LINK_CONNECT_TEMPLATE(1);
	SIGNAL_LINK_CONNECT_TEMPLATE(2);
	SIGNAL_LINK_CONNECT_TEMPLATE(3);
	SIGNAL_LINK_CONNECT_TEMPLATE(4);
	SIGNAL_LINK_CONNECT_TEMPLATE(5);
	SIGNAL_LINK_CONNECT_TEMPLATE(6);
	SIGNAL_LINK_CONNECT_TEMPLATE(7);
	SIGNAL_LINK_CONNECT_TEMPLATE(8);
	SIGNAL_LINK_CONNECT_TEMPLATE(9);
	
	template<typename T, typename Signature> void signal_link_connect(boost::signal<Signature> &signal, T &link)
	{
		signal_link_connect_impl<T, boost::function_traits<Signature>::arity, Signature>::connect(signal, link);
	}

	template<typename T, typename Signature> void signal_link_connect_slot(boost::signal<Signature> &signal, const slot_selector_t<T, Signature> &link)
	{
		signal_link_connect_impl<T, boost::function_traits<Signature>::arity, Signature>::connect_slot(signal, link);
	}
}

#undef SIGNAL_LINK_ARG
#undef SIGNAL_LINK_ARGS
#undef SIGNAL_LINK_CAST
#undef SIGNAL_LINK_CAST_COMMA
#undef SIGNAL_LINK_CASTS
#undef SIGNAL_LINK_TEMPLATE

///	Provides easy linking of components through Boost signals.
/**	\ingroup Utility
	\param Descendant The name of the class subclassing signal_link.
	\param Signature The signature of the signal being sent out.

	Use this class as a base class for classes that produce a \e signal
	of a particular signature.  Classes that contain operator() with
	the same signature will be able to receive those signals.
	Multiple objects can be connected to receive the same signal.

	Signals are sent using the Boost.Signals (http://www.boost.org/doc/html/signals.html)
	You can view the associated tutorial (http://www.boost.org/doc/html/signals/tutorial.html)
	to get familiar with how it works.  Using signal_link
	as a base class for signal sending, signals can be received by any
	class using the operator() of the appropriate signature.  Such receivers
	are called \e slots.

\section signal_link_examples Examples
	\li \ref signal_link_simple_example
	\li \ref signal_link_branching_example
	\li \ref signal_link_disconnect_example
	\li \ref signal_link_multi_in_simple_example
	\li \ref signal_link_multi_in_example

\subsection signal_link_simple_example Simple connections

	For example, consider the signature "void ()".  This is the signature
	of a function that returns void and takes no arguments.
	A class that can send out signals of such a signature would be
	defined as follows:

	\dontinclude SignalLinkTests.cpp
	\skip class SignalVoid
	\until end class SignalVoid

	Signals of the same signature could be received by
	the following class:

	\skip class SignalVoidCounter
	\until end class SignalVoidCounter

	The following demonstrates how you would make a signal sender / receiver
	where the signal carried a float argument:

	\skip class SignalFloat
	\until end class SignalFloatCollector

	The SignalFloat class above can also receive signals
	of signature void(), because it contains a void operator()() definition.

	Using the above classes, it is easy to connect them together using the
	signal_link::operator>>=:
	\dontinclude SignalLinkTests.cpp
	\skip simple_test
	\until end void simple_test

\subsection signal_link_branching_example Branching connections
	More complex connections can also be made relatively easily using
	both signal_link::operator>>= and signal_link::operator>=.  The
	following code will result in the same final signal network as
	the \ref signal_link_simple_example example above:

	\dontinclude SignalLinkTests.cpp
	\skip branching_test
	\until end void branching_test

\subsection	signal_link_disconnect_example Disconnecting connections
	Connections can be terminated in two ways.  One is through the "trackable"
	mechanism of Boost.Signals, which will automatically destroy connections
	to a trackable object when the object is destroyed.  The other
	way is through the disconnect_all_slots method of the signal sender.

	\dontinclude SignalLinkTests.cpp
	\skip void disconnect_test
	\until end void disconnect_test

\subsection signal_link_multi_in_simple_example Multiple inputs of different signatures

	It is simple to have an object provide multiple slots through operator() functions
	of different signatures.  The following class does so through providing 1-ary slots
	of different types:

	\dontinclude SignalLinkTests.cpp
	\skip class SignalIntFloatCollector
	\until end class

	The following class, on the other hand, uses slots of different number of arguments:
	\dontinclude SignalLinkTests.cpp
	\skip class SignalMultiCollector
	\until end class

	In such cases, where the operator() functions differ in their signature,
	standard connection operators will work out the correct connection:

	\dontinclude SignalLinkTests.cpp
	\skip void multi_type_test
	\until end void multi_type_test

\subsection signal_link_multi_in_example Multiple inputs of the same signature

	In some cases, a class may want to receive multiple signals of the same
	signature.  For example, the following class can receive a void() signal
	through its inherited operator() function, as well as through the
	operator() function of member "other":
	\dontinclude SignalLinkTests.cpp
	\skip class Signal2VoidCounter
	\until end class
	
	Similarly, the following class could receive void() signals both
	through operator() and through AltInput:
	\dontinclude SignalLinkTests.cpp
	\skip class Signal2VoidInputs
	\until end class

	The following example shows how to connect signals to all of the above slots.
	For the class Signal2VoidInputs, this is accomplished using the
	slot_selector function:
	\skip void multi_in_test
	\until end void multi_in_test

*/
template<typename Descendant, typename Signature>
class signal_link : public boost::signals::trackable
{
public:
	typedef boost::signal<Signature> default_signal_t;
	signal_link(const signal_link &) {}
	signal_link(){}
	///	Returns the default out signal.
	const default_signal_t &default_signal() const
	{
		return out;
	}
	///	Connects a sequence of components using signals.
	/** This operator is identical to signal_link::operator>= (it connects the
	left component to the right component, and returns a reference to the left component),
	except it is evaluated right to left.  This makes it semantics more suitable for
	connecting a chain of connections.
	*/
	template<typename T>
	Descendant &operator >>= (T &link) {detail::signal_link_connect(out, link); return *(Descendant *)this;}
	/// Allows branching in a component connection sequence.
	/** This operator is identical to signal_link::operator>>=, (it connects the
	left component to the right component, and returns a reference to the left component)
	except it is evaluated left to right.  This makes its semantics more suitable for
	branching connections.
	*/
	template<typename T>
	Descendant &operator >= (T &link) {detail::signal_link_connect(out, link); return *(Descendant *)this;}
	///	Allows slot functions other than operator() to be used in a sequence of components.
	/**	\sa slot_selector()
	*/
	template<typename T>
	Descendant &operator >>= (slot_selector_t<T, Signature> link) {detail::signal_link_connect(out, link); return *(Descendant *)this;}
	/// Allows slot functions other than operator() to be used with branching.
	/** \sa slot_selector()
	*/
	template<typename T>
	Descendant &operator >= (const slot_selector_t<T, Signature> &link) {detail::signal_link_connect_slot(out, link); return *(Descendant *)this;}
	///	Disconnects all slots connected to the signal_link.
	void disconnect_all_slots() {out.disconnect_all_slots();}
protected:
	default_signal_t out;
};

template<typename Signature, typename T>
boost::signal<Signature> &operator >>= (boost::signal<Signature> &signal, T &link)
{detail::signal_link_connect(signal, link); return signal;}

template<typename Signature, typename T>
boost::signal<Signature> &operator >= (boost::signal<Signature> &signal, T &link)
{detail::signal_link_connect(signal, link); return signal;}	

}

#endif // AME_SIGNAL_LINK_HPP