
#ifndef ASIO_BASIC_STREAM_FILE_HPP
#define ASIO_BASIC_STREAM_FILE_HPP

#include "asio/detail/push_options.hpp"

#include <boost/noncopyable.hpp>

#include "asio/default_error_handler.hpp"
#include "asio/service_factory.hpp"

#include <boost/static_assert.hpp>

namespace asio {

template <typename Service>
class basic_stream_file
  : private boost::noncopyable
{
public:
  // The type of the service that will be used to provide file operations.
  typedef Service service_type;

  // The native implementation type of the stream socket.
  typedef typename service_type::impl_type impl_type;
  
  // The demuxer type for this asynchronous type.
  typedef typename service_type::demuxer_type demuxer_type;

  // A basic_stream_file is always the lowerst layer
  typedef basic_stream_file<service_type> lowest_layer_type;

  explicit basic_stream_file(demuxer_type& d)
    : service_(d.get_service(service_factory<Service>())),
      impl_(service_type::null())
  {}
  
  basic_stream_file(demuxer_type& d, const char* name, int flags)
    : service_(d.get_service(service_factory<Service>())),
      impl_(open(name, flags))
  {}

  ~basic_stream_file()
  {
    service_.close(impl_);
  }

  /// Get the demuxer associated with the asynchronous object.
  demuxer_type& demuxer()
  {
    return service_.demuxer();
  }

  void close()
  {
    service_.close(impl_);
  }

  /// Get a reference to the lowest layer.
  lowest_layer_type& lowest_layer()
  {
    return *this;
  }

  /// Get the underlying implementation in the native type.
  impl_type impl()
  {
    return impl_;
  }

  /// Set the underlying implementation in the native type.
  void set_impl(impl_type new_impl)
  {
    service_.open(impl_, new_impl);
  }

  /// Start an asynchronous write.
  template <typename Handler>
  void async_write(const void* data, size_t length, long long offset, Handler handler)
  {
    service_.async_write(impl_, data, length, offset, handler);
  }

  size_t write(const void* data, size_t length)
  {
    return service_.send(impl_, data, length, default_error_handler());
  }

  template <typename ErrorHandlerT>
  size_t write(const void* data, size_t length, ErrorHandlerT error_handler)
  {
    return service_.send(impl_, data, length, error_handler);
  }

  template <typename Handler>
  void async_read(void* data, size_t max_length, long long offset, Handler handler)
  {
    service_.async_read(impl_, data, max_length, offset, handler);
  }

  size_t read(const void* data, size_t max_length)
  {
    return service_.read(impl_, data, max_length, default_error_handler());
  }

  template <typename ErrorHandlerT>
  size_t read(const void* data, size_t max_length, ErrorHandlerT error_handler)
  {
    return service_.read(impl_, data, max_length, error_handler);
  }

  // size_t in_avail(ErrorHandlerT error_handler)

private:
  /// The backend service implementation
  service_type& service_;

  /// The underlying native implementation
  impl_type impl_;

};

} // namespace asio

#endif
