
#ifndef ASIO_DETAIL_AIO_DEMUXER_SERVICE_HPP
#define ASIO_DETAIL_AIO_DEMUXER_SERVICE_HPP

#include "asio/detail/aio_operation.hpp"

#include <libaio.h>

#ifndef ASIO_AIO_MAX_FILEDESCRIPTORS
#define ASIO_AIO_MAX_FILEDESCRIPTORS 512
#endif

namespace asio {
namespace detail {

class aio_demuxer_service
{
public:
  typedef basic_demuxer<aio_demuxer_service> demuxer_type;

  // constructor
  aio_demuxer_service(demuxer_type& demuxer)
    : demuxer_(demuxer),
      outstanding_work_(0),
      interrupted_(false)
  {
    memset(reinterpret_cast<void*>(&io_context_), 0, sizeof(io_context_));

    if(::io_queue_init(ASIO_AIO_MAX_FILEDESCRIPTORS, &io_context_) != 0) {
      throw std::bad_alloc();
    }
  }

  ~aio_demuxer_service()
  {
    io_queue_release(io_context_);
  }

  void run()
  {
    while(true) {
      if(interrupted_)
        return;

      struct io_event events[128];
      struct timespec timeout = { 1, 0 }; // one second
      long num_events = io_getevents(io_context_, 1, sizeof(events)/sizeof(events[0]), events, &timeout);
      for(long i = 0; i < num_events; ++i) {
        aio_operation* op = static_cast<aio_operation*>(events[i].obj);
        if(op->aio_lio_opcode == IO_CMD_PREAD || op->aio_lio_opcode == IO_CMD_PWRITE) {
          // didnt read all that should
          if(events[i].res != op->u.c.nbytes) {
            struct stat s;
            fstat(op->aio_fildes, &s);
            // if we're reading, and the file is bigger than we asked for
            // then an error happened
            if((op->aio_lio_opcode == IO_CMD_PREAD && s.st_size > (op->u.c.nbytes + op->u.c.offset))
               || (static_cast<int>(events[i].res)) < 0) {
              asio::error error(-static_cast<int>(events[i].res));
              op->do_completion(error, 0);
              continue;
            }
          }
          asio::error error(0);
          work_finished();
          op->do_completion(error, events[i].res);
        }
        else {
          assert(false && "Not an IO_CMD_PREAD nor IO_CMD_PWRITE aio event");
        }
      }
    }
  }

  void work_started()
  {
    // increment outstanding_work_
    asio::detail::mutex::scoped_lock lock(mutex_);
    outstanding_work_++;
  }

  void work_finished()
  {
    // decrement outstanding_work_
    // if outstanding_work becomes zero, call interrupt
    asio::detail::mutex::scoped_lock lock(mutex_);
    if(!(--outstanding_work_))
      interrupt();
  }

  // find out how to post
  template <typename Handler>
  void post(Handler handler)
  {
    
  }

  // interrupts the run function
  void interrupt()
  {
    interrupted_ = true;
  }

  void reset()
  {
    interrupted_ = false;
  }

private:
  friend class aio_stream_file_service;

  io_context_t io_context()
  {
    return io_context_;
  }

  demuxer_type& demuxer_;
  int outstanding_work_;
  asio::detail::mutex mutex_;
  bool interrupted_;

  io_context_t io_context_;
};

}
}

#endif
