
#ifndef ASIO_DETAIL_AIO_STREAM_FILE_SERVICE_HPP
#define ASIO_DETAIL_AIO_STREAM_FILE_SERVICE_HPP

#include "asio/detail/push_options.hpp"

#include "asio/error.hpp"
#include "asio/detail/file_types.hpp"
#include "asio/detail/aio_demuxer_service.hpp"

#include <libaio.h>

namespace asio {
namespace detail {

class aio_stream_file_service
{
public:
  typedef file_type impl_type;

  static impl_type null()
  {
    return invalid_file;
  }

  typedef basic_demuxer<aio_demuxer_service> demuxer_type;

  aio_stream_file_service(demuxer_type& demuxer)
    : demuxer_(demuxer),
      demuxer_service_(demuxer.get_service(
          service_factory<aio_demuxer_service>()))
  {
  }

  demuxer_type& demuxer()
  {
    return demuxer_;
  }

  void open(impl_type& impl, impl_type new_impl)
  {
    impl = new_impl;
  }

  void close(impl_type& impl)
  {
    if(impl != null()) {
      ::close(impl);
      impl = null();
    }
  }

  template <typename ErrorHandlerT>
  size_t write(impl_type& impl, const void* data, size_t length,
               ErrorHandlerT error_handler)
  {
    int bytes_writed = ::write(impl, data, length);
    if(bytes_writed < 0) {
      error_handler(asio::error(errno));
      return 0;
    }
    return bytes_writed;
  }

  template <typename ErrorHandlerT>
  size_t read(impl_type& impl, const void* data, size_t max_length,
              ErrorHandlerT error_handler)
  {
    int bytes_readed = ::read(impl, data, length);
    if(bytes_readed < 0) {
      error_handler(asio::error(errno));
      return 0;
    }
    return bytes_readed;
  }

  template <typename Handler>
  class completed_operation
    : public aio_operation
  {
    Handler handler_;
  public:
    completed_operation(Handler handler) : handler_(handler) {}
    ~completed_operation() {}

    void do_completion(asio::error error, size_t nbytes)
    {
      handler_(error, nbytes);
    }
  };

  // Starts an asynchronous write. The data being writed must be valid for the
  // lifetime of the asynchronous opereation. (This function was "copied" from
  // the reactive_stream_socket_service::async_send
  template <typename Handler>
  void async_write(impl_type& impl, const void* data, size_t length, long long offset,
                   Handler handler)
  {
    if (impl == null()) {
      asio::error error(asio::error::bad_descriptor);
      demuxer_.post(bind_handler(handler, error, 0));
    }
    else {
      completed_operation<Handler> *op = new completed_operation<Handler>(handler);

      demuxer_service_.work_started();

      struct iocb* p = op;
      
      ::io_prep_pread(op, impl, const_cast<void*>(data), length, offset);
      if(::io_submit(demuxer_service_.io_context(), 1, &p) != 1) {
        delete op;
        asio::error error(errno);
        demuxer_service_.post(bind_handler(handler, error, 0));
        demuxer_service_.work_finished();
      }
    }
  }

  // Start an asynchronous read. The buffer for the data being readed
  // must be valid for the lifetime of the asynchronous operation.
  // (This function was "copied" from the
  // reactive_stream_socket_service::async_read)
  template <typename Handler>
  void async_read(impl_type& impl, void* data, size_t max_length, long long offset,
                  Handler handler)
  {
    if (impl == null()) {
      asio::error error(asio::error::bad_descriptor);
      demuxer_service_.post(bind_handler(handler, error, 0));
    }
    else {
      completed_operation<Handler> *op = new completed_operation<Handler>(handler);

      demuxer_service_.work_started();

      struct iocb* p = op;
      
      ::io_prep_pread(op, impl, data, max_length, offset);
      if(::io_submit(demuxer_service_.io_context(), 1, &p) != 1) {
        delete op;
        asio::error error(errno);
        demuxer_service_.post(bind_handler(handler, error, 0));
        demuxer_service_.work_finished();
      }
    }
  }

  // size_t in_avail(impl_type& impl, ErrorHandlerT error_handler)

private:
  // The demuxer used for dispatching handlers
  demuxer_type& demuxer_;

  // The demuxer service used for running asynchronous operations
  // and dispatching handlers.
  aio_demuxer_service& demuxer_service_;
};

} }

#endif

